// T1Store.h

/*	Usage Instructions:
 
 Drag the T1Store folder into XCode, and be sure to copy everything when prompted.
 
 To show the store programmatically:
 
 1) #import "T1Store.h" in your view controller.h
 2) [[T1Store sharedT1Store] show:self];
 
 
 To show the store using interface builder:
 
 1) #import "T1Store.h" in your view controller.h
 2) add [T1Store version]; to your view controller init method (to ensure the library is linked when you compile) drag the T1Store class into your interface builder window (or drag in a blank object, and give it a class identity of T1Store) right-click on a button, and drag a connection to the show: method of the T1Store object
 
 
 There are a few nice customizations and options listed below. Be sure to check them out tintColor for iOS7 and showForViewController:withProducts:inCart:allowOthers: method to put product directly in your customer's cart, reducing the number of steps to purchase.  You may also set a delegate and listen for events from the store.
 */

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
@class T1Store;


@protocol T1StoreDelegate <NSObject>
@optional
- (void)storeDidDismissModalView:(T1Store *)store;	// user closed store view
- (BOOL)store:(T1Store *)store shouldFailWithError:(NSError *)error;	// connection could not be made to store.  Return NO to hide the built-in alert error message if you'd like to present your own
- (void)store:(T1Store *)store didCompletePurchase:(NSDictionary *)purchase;	// dictionary containing purchase details. Do NSLog(@"%@",purchase) to see the available data after a simulated purchase using testMode.
@end





@interface T1Store : NSObject <UITabBarControllerDelegate>

@property (weak) id <T1StoreDelegate> delegate;	// See T1StoreDelegate protocol above
@property (assign, readonly) BOOL visible;		// store view is visible
@property (strong, readonly) NSString *storeVersion;	// version of this library
@property (strong) NSString *storeTitle;	// Default: Ten One Design.  Keep it short for iPhone
@property (strong) NSString *storeFocus;	// @"all" (default), @"stylus", @"pogo connect", or @"gaming".
@property (strong) UIColor *tintColor;		// Control over tint of buttons. Default: nil (system color)
@property (assign) BOOL testMode;			// If YES, skips paypal flow and simulates a purchase
@property (assign) float developerEarningRatio;	// Default: 0.5.  This is a float between 0 and 1 describing ratio of money kept by developer.  Calculation: (Developer-kept % of total sale)/(total discount offered by Ten One).  Example:  If Ten One is offering a 20% total discount, a ratio of 0.5 will give 10% of purchase price to developer, and pass on 10% savings to the customer.  A value of 1.0 will result in the developer keeping a full 20%, while 0.2 will result in the developer keeping only 4%.  This value may be changed at any time at your discretion, and can vary for each transaction.




+ (T1Store *)sharedT1Store;	// primary singleton access


- (IBAction)show:(id)sender;	// present store from the first view controller in the view heirarchy
- (IBAction)showForViewController:(id)startingViewController;	// presents itself from a specified view controller
- (IBAction)showForViewController:(id)startingViewController withProducts:(NSArray *)productArray inCart:(BOOL)inCart allowOthers:(BOOL)allowOthers;	// useful to skip store browswer and place products directly in the cart.  partNumberStringArray is an NSArray of NSDictionaries containing @"number" and @"quantity" keys.  Contact info@tenonedesign.com to obtain the latest part numbers to use for the number value. A good example is in the demo project.


- (IBAction)hide:(id)sender;	// can be called at any time to hide store


- (IBAction)freeMemory:(id)sender;	// deallocates memory, but only if store is hidden
- (IBAction)reset:(id)sender;	// clears memory, hides, and initializes the store


@end






